/****************************************************************************
*
*    Copyright (c) 2005 - 2014 by Vivante Corp.  All rights reserved.
*
*    The material in this file is confidential and contains trade secrets
*    of Vivante Corporation. This is proprietary information owned by
*    Vivante Corporation. No part of this work may be disclosed,
*    reproduced, copied, transmitted, or used in any way for any purpose,
*    without the express written permission of Vivante Corporation.
*
*****************************************************************************/


/*!
 @header tinyUI API
 The API programs can use to use the tinyUI accelerator.

 @copyright Vivante Corporation
 */

#ifndef _tiny_ui_h_
#define _tiny_ui_h_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include "tiny_ui_error.h"

#if 0
#pragma mark Types
#endif

/*!
 @abstract The floating point type used by the tinyUI API.
 */
typedef float tiny_ui_float_t;

/*!
 @abstract A 32-bit color value used by the tinyUI API.

 @discussion
 The color value specifies the color used in various functions. The color is formed using 8-bit RGBA channels. The red channel
 is in the lower 8-bit of the color value, followed by the green and blue channels. The alpha channel is in the upper 8-bit of
 the color value.

 For L8 target formats, the RGB color is converted to L8 by using the default ITU-R BT.709 conversion rules.
 */
typedef uint32_t tiny_ui_color_t;

#if 0
#pragma mark Enumerations
#endif

/*!
 @abstract Quality enumeration for a given path.

 @discussion
 Each path should specify a quality hint for the hardware. The path generation tool will generate the quality hint based on the
 complexity of the path.
 */
typedef enum tiny_ui_quality {
    TINY_UI_HIGH,   /*! High quality 16x anti-aliasing path. */
    TINY_UI_MEDIUM, /*! Medium quality 4x anti-aliasing path. */
    TINY_UI_LOW,    /*! Low quality pat without any anti-aliasing. */
} tiny_ui_quality_t;

/*!
 @abstract Format of path coordinates.

 @discussion
 Each path can have a separate coordinate system. The path generation tool will find the most optimal coordinate system for any
 given path based on its dimensions and input coordinates.
 */
typedef enum tiny_ui_format {
    TINY_UI_S8,     /*! Signed 8-bit coordinates. */
    TINY_UI_S16,    /*! Signed 16-bit coordinates. */
    TINY_UI_S32,    /*! Signed 32-bit coordinates. */
    TINY_UI_FP32,   /*! 32-bit floating point coordinates. */
} tiny_ui_format_t;

/*!
 @abstract Format of a buffer.

 @discussion
 The pixel type for a <code>tiny_ui_buffer_t</code> structure.
 */
typedef enum tiny_ui_buffer_format {
    TINY_UI_RGBA8888,   /*! 32-bit RGBA format with 8 bits per color channel. Red is in bits 7:0, green in bits 15:8, blue in
                         bits 23:16, and the alpha channel is in bits 31:24. */
    TINY_UI_BGRA8888,   /*! 32-bit RGBA format with 8 bits per color channel. Red is in bits 23:16, green in bits 15:8, blue in
                         bits 7:0, and the alpha channel is in bits 31:24. */
    TINY_UI_RGB565,     /*! 16-bit RGB format with 5 and 6 bits per color channel. Red is in bits 4:0, green in bits 10:5, and
                         the blue color channel is in bits 15:11. */
    TINY_UI_BGR565,     /*! 16-bit RGB format with 5 and 6 bits per color channel. Red is in bits 15:11, green in bits 10:5,
                         and the blue color channel is in bits 4:0. */
    TINY_UI_RGBA4444,   /*! 16-bit RGBA format with 4 bits per color channel. Red is in bits 3:0, green in bits 7:4, blue in
                         bits 11:8 and the alpha channel is in bits 15:12. */
    TINY_UI_BGRA4444,   /*! 16-bit RGBA format with 4 bits per color channel. Red is in bits 11:8, green in bits 7:4, blue in
                         bits 3:0 and the alpha channel is in bits 15:12. */
    TINY_UI_A8,         /*! 8-bit alpha format. There are no RGB values. */
    TINY_UI_L8,         /*! 8-bit luminance value. There is no alpha value. */
    TINY_UI_YUYV,       /*! Packed YUV format, 32-bit for 2 pixels. Y0 is in bits 7:0 and V is in bits 31:23. */
} tiny_ui_buffer_format_t;

/*!
 @abstract Blending modes.

 @discussion
 Some of the tinyUI API functions calls support blending. S and D represent source and destination color channels and Sa and Da
 represent the source and destination alpha channels.
 */
typedef enum tiny_ui_blend {
    TINY_UI_BLEND_NONE,     /*! S, i.e. no blending. */
    TINY_UI_BLEND_SRC_OVER, /*! S + (1 - Sa) * D */
    TINY_UI_BLEND_DST_OVER, /*! (1 - Da) * S + D */
    TINY_UI_BLEND_SRC_IN,   /*! Da * S */
    TINY_UI_BLEND_DST_IN,   /*! Sa * D */
    TINY_UI_BLEND_SCREEN,   /*! S + D - S * D */
    TINY_UI_BLEND_MULTIPLY, /*! S * (1 - Da) + D * (1 - Sa) + S * D */
    TINY_UI_BLEND_ADDITIVE, /*! S + D */
    TINY_UI_BLEND_SUBTRACT, /*! D * (1 - S) */
} tiny_ui_blend_t;

/*!
 @abstract Fill rules.

 @discussion
 For drawing any path, the hardware supports both non-zero and odd-even fill rules.

 To determine whether any point is contained inside an object, imagine drawing a line from that point out to infinity in any
 direction such that the line does not cross any vertex of the path. For each edge that is crossed by the line, add 1 to the
 counter if the edge crosses from left to right, as seen by an observer walking across the line towards infinity, and subtract 1
 if the edge crosses from right to left. In this way, each region of the plane will receive an integer value.

 The non-zero fill rule says that a point is inside the shape if the resulting sum is not equal to zero. The even/odd rule says
 that a point is inside the shape if the resulting sum is odd, regardless of sign.
 */
typedef enum tiny_ui_fill {
    TINY_UI_FILL_NON_ZERO,  /*! Non-zero fill rule. A pixel is drawn if it crosses at least one path pixel. */
    TINY_UI_FILL_EVEN_ODD,  /*! Even-odd fill rule. A pixel is drawn it it crosses an odd number of path pixels. */
} tiny_ui_fill_t;

#if 0
#pragma mark Structures
#endif

/*!
 @abstract A 3x3 matrix.

 @discussion
 For those functions that need a matrix, this is the structure that defines it. The contents are a simple 3x3 matrix
 consisting of floating pointer numbers.
 */
typedef struct tiny_ui_matrix {
    tiny_ui_float_t m[3][3];    /*! The 3x3 matrix itself, in [row][column] order. */
} tiny_ui_matrix_t;

/*!
 @abstract A wrapper structure for any image or render target.

 @discussion
 Each piece of memory, whether it is an image used as a source or a buffer used as a target, requires a structure to define it.
 This structure contains all the information the tinyUI API requires to access the buffer's memory by the hardware.
 */
typedef struct tiny_ui_buffer {
    int32_t width;                  /*! Width of the buffer in pixels. */
    int32_t height;                 /*! Height of the buffer in pixels. */
    int32_t stride;                 /*! The number of bytes to move from one line in the buffer to the next line. */
    tiny_ui_buffer_format_t format; /*! The pixel format of the buffer. */
    void * handle;                  /*! The memory handle of the buffer's memory as allocated by the tinyUI kernel. */
    void * memory;                  /*! The logical pointer to the buffer's memory for the CPU. */
    uint32_t address;               /*! The address to the buffer's memory for the hardware. */
} tiny_ui_buffer_t;

/*!
 @abstract A path used by the drawing command.

 @discussion
 Each path needs a few parameters. This structure defines those parameters, so the tinyUI driver knows the detail of a path.
 */
typedef struct tiny_ui_path {
    tiny_ui_float_t bounding_box[4];    /*! Bounding box specified as left, top, right, and bottom. */
    tiny_ui_quality_t quality;          /*! Quality hint for the path. */
    tiny_ui_format_t format;            /*! Coordinate format. */
    tiny_ui_buffer_t * uploaded;        /*! Pointer to a <code>tiny_ui_buffer_t</code> structure if the path data has been
                                         upload into GPU addressable memory. */
    int32_t path_length;                /*! Number of bytes in the path data. */
    void * path;                        /*! Pointer to the physical description of the path. */
} tiny_ui_path_t;

/*!
 @abstract A rectangle.

 @discussion
 A rectangle defines a rectangular definition of the screen.
 */
typedef struct tiny_ui_rectangle {
    int32_t x;      /*! Left coordinate of the rectangle. */
    int32_t y;      /*! Top coordinate of the rectangle. */
    int32_t width;  /*! Width of the rectangle. */
    int32_t height; /*! Height of the rectangle. */
} tiny_ui_rectangle_t;

#if 0
#pragma mark Function prototypes.
#endif

/*!
 @abstract Allocate a buffer from hardware accessible memory.

 @discussion
 In order for the hardware to access some memory, like a source image or a target buffer, it needs to be allocated first. The
 supplied <code>tiny_ui_buffer_t</code> structure needs to be initialized with the size (width and height) and format of the
 requested buffer. If the stride is set to zero, this function will fill it in.

 This function will call the kernel to actually allocate the memory and the memory handle and logical and hardware addresses
 will be filled in by the kernel.

 @param buffer
 Pointer to the buffer that holds the size and format of the buffer being allocated.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_allocate(tiny_ui_buffer_t * buffer);

/*!
 @abstract Free a buffer that was previously allocated by {@link tiny_ui_allocate}.

 @discussion
 Free any memory resources allocated by a previous call to {@link tiny_ui_allocate}.

 @param buffer
 Pointer to a buffer structure that was filled in by {@link tiny_ui_allocate}.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_free(tiny_ui_buffer_t * buffer);

/*!
 @abstract Map a buffer into hardware accessible address space.

 @discussion
 If you want the use a frame buffer directly as an target buffer, you need to wrap a <code>tiny_ui_buffer_t</code> structure
 around it and call the kernel to map the supplied logical or physical address into hardware accessible memory.

 For example, if you know the logical address of the frame buffer, set the memory field of the tiny_ui_buffer_t structure
 with that address and call this function. If you know the physical address, set the memory field to <code>NULL</code> and
 program the address field with the physical address.

 @param buffer
 Pointer to the buffer that holds the size and format of the buffer being allocated. Either the memory or address field
 needs to be set to a non-zero value to map either a logical or physical address into hardware accessible memory.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_map(tiny_ui_buffer_t * buffer);

/*!
 @abstract Unmap a buffer that was previously mapped by {@link tiny_ui_map}.

 @discussion
 Free any memory resources allocated by a previous call to {@link tiny_ui_map}.

 @param buffer
 Pointer to a buffer structure that was filled in by {@link tiny_ui_map}.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_unmap(tiny_ui_buffer_t * buffer);

/*!
 @abstract Initialize a blit context.

 @discussion
 The {@link tiny_ui_clear} and {@link tiny_ui_blit} functions require a blit context to be initialized. There is only one blit
 context per process, so this function has be called once in your application. If this would be the first context that accesses
 the hardware, the hardware will be turned on and initialized.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_blit_init(void);

/*!
 @abstract Destroy a blit context.

 @discussion
 Destroy a blit context that was previously initialized by {@link tiny_ui_blit_init}.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_blit_close(void);

/*!
 @abstract Fill a (partial) buffer with a specified color.

 @discussion
 Either an entire buffer or a partial rectangle of a buffer will be filled with a specific color.

 This function will wait until the hardware is complete, i.e. it is synchronous.

 @param target
 Pointer to a <code>tiny_ui_buffer_t</code> structure that describes the buffer to be filled.

 @param rectangle
 Pointer to a rectangle that specifies the area to be filled. If <code>rectangle</code> is <code>NULL</code>, the entire target
 buffer will be filled with the specified color.

 @param color
 The color value to use for filling the buffer. If the buffer is in L8 format, the RGBA color will be converted into a
 luminance value.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_clear(tiny_ui_buffer_t * target,
                              tiny_ui_rectangle_t * rectangle,
                              tiny_ui_color_t color);


/*!
 @abstract Copy a source image to the the destination window with a specified matrix that can include translation, rotation,
 scaling, and perspective correction.

 @discussion
 A source image is copied to the target using the specified matrix. If the specified matrix is <code>NULL</code>, an identity
 matrix is assumed, meaning the source will be copied directly on the target at 0,0 location.

 An optional blend mode can be specified that defines the blending of the source onto the target.

 Also, an optional mix color can be specified. The mix color will be multiplied by the source color. If you don't need a mix
 color, set the <code>color</code> parameter to 0.

 Note that on hardware that doesn't support border scissoring (GC355) the blend mode will be forced to
 <code>TINY_UI_BLEND_SRC_OVER</code> if rotation or perspective is involved.

 @param target
 Pointer to a <code>tiny_ui_buffer_t</code> structure that describes the target of the blit.

 @param source
 Pointer to a <code>tiny_ui_buffer_t</code> structure that describes the source of the blit.

 @param matrix
 Pointer to a 3x3 matrix that defines the transformation matrix of source pixels into the target. If matrix is
 <code>NULL</code>, an identity matrix is assumed.

 @param blend
 The blending mode to be applied to each image pixel. If no blending is required, set this value to
 <code>TINY_UI_BLEND_NONE</code> (0).

 @param color
 If non-zero, this color value will be used as a mix color. The mix color gets multiplied with each source pixel before
 blending happens.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_blit(tiny_ui_buffer_t * target,
                             tiny_ui_buffer_t * source,
                             tiny_ui_matrix_t * matrix,
                             tiny_ui_blend_t blend,
                             tiny_ui_color_t color);


/*!
 @abstract Initialize a draw context.

 @discussion
 The {@link tiny_ui_draw} function requires a draw context to be initialized. There is only one draw context per process, so
 this function has be called once in your application if any draw command will be used. If this would be the first context that
 accesses the hardware, the hardware will be turned on and initialized.

 The difference between a blit and draw context is that the draw context has a larger command buffer and allocates a
 tessellation buffer for the hardware. The size of the tessellation buffer can be specified, and that size will be aligned to
 the minimum required alignment of the hardware by the kernel. If you make the tessellation buffer smaller, less memory will
 be allocated, but a path might be sent down to the hardware multiple times because the hardware will walk the target with the
 provided tessellation window size, so performance might go down. It is good practice to set the tessellation buffer size to the
 most common path size. For example, if all you do is render up to 24-pt fonts, you can set the tessellation buffer to be
 24x24.

 @param tessellation_width
 The width of the tessellation window.

 @param tessellation_height
 The height of the tessellation window.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_draw_init(int32_t tessellation_width,
                                  int32_t tessellation_height);

/*!
 @abstract Destroy a draw context.

 @discussion
 Destroy a draw context that was previously initialized by {@link tiny_ui_draw_init}.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_draw_close(void);

/*!
 @abstract Draw a path to a target buffer.

 @discussion
 The specified path will be transformed by the given matrix and drawn into the specified target buffer using the supplied color.
 Blending can be specified.

 @param target
 Pointer to a <code>tiny_ui_buffer_t</code> structure that describes the target of the draw.

 @param path
 Pointer to a <code>tiny_ui_path_t</code> structure that describes the path to draw.

 @param fill_rule
 Specified fill rule for the path.

 @param matrix
 Pointer to a 3x3 matrix that defines the transformation matrix of the path. If <code>matrix</code> is <code>NULL</code>, an
 identity matrix is assumed which is usually a bad idea since the path can be anything.

 @param blend
 The blending mode to be applied to each drawn pixel. If no blending is required, set this value to
 <code>TINY_UI_BLEND_NONE</code> (0).

 @param color
 The color applied to each pixel drawn by the path.

 @result
 Returns the status as defined by <code>tiny_ui_error_t</code>.
 */
tiny_ui_error_t tiny_ui_draw(tiny_ui_buffer_t * target,
                             tiny_ui_path_t * path,
                             tiny_ui_fill_t fill_rule,
                             tiny_ui_matrix_t * matrix,
                             tiny_ui_blend_t blend,
                             tiny_ui_color_t color);

#ifdef __cplusplus
}
#endif
#endif /* _tiny_ui_h_ */
